const { z } = require('zod');
const prisma = require('../prisma');
const { verifyPassword } = require('../utils/crypto');
const { signToken } = require('../utils/jwt');
const { updateSettings, getSettings, updateSiteSettings, getPublicSettings } = require('../services/settingsService');
const { syncGames } = require('../services/gameService');

const loginSchema = z.object({
  email: z.string().email(),
  password: z.string().min(6),
});

const settingsSchema = z.object({
  product_margin_percent: z.number().nonnegative(),
  deposit_fee_percent: z.number().nonnegative(),
});

const siteSchema = z.object({
  site_name: z.string().min(1).optional(),
  site_description: z.string().optional(),
  site_keywords: z.string().optional(),
  site_logo_url: z.string().optional(),
  site_favicon_url: z.string().optional(),
  site_og_image_url: z.string().optional(),
  cs_link: z.string().optional(),
  footer_text: z.string().optional(),
  maintenance_mode: z.enum(['none', 'admin', 'developer']).optional(),
});

const gameStatusSchema = z.object({
  enabled: z.boolean(),
});

async function login(req, res, next) {
  try {
    const payload = loginSchema.parse(req.body);
    const admin = await prisma.adminUser.findUnique({ where: { email: payload.email } });
    if (!admin) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }
    const ok = await verifyPassword(payload.password, admin.passwordHash);
    if (!ok) {
      return res.status(401).json({ message: 'Invalid credentials' });
    }
    const token = signToken({ sub: admin.id, admin: true });
    res.json({ token, admin: { id: admin.id, email: admin.email, role: admin.role } });
  } catch (err) {
    next(err);
  }
}

async function getSettingsHandler(req, res, next) {
  try {
    const settings = await getSettings();
    res.json({
      product_margin_percent: Number(settings.productMarginPercent),
      deposit_fee_percent: Number(settings.depositFeePercent),
    });
  } catch (err) {
    next(err);
  }
}

async function updateSettingsHandler(req, res, next) {
  try {
    const payload = settingsSchema.parse(req.body);
    const settings = await updateSettings({
      productMarginPercent: payload.product_margin_percent,
      depositFeePercent: payload.deposit_fee_percent,
    });
    res.json({
      product_margin_percent: Number(settings.productMarginPercent),
      deposit_fee_percent: Number(settings.depositFeePercent),
    });
  } catch (err) {
    next(err);
  }
}

async function syncGamesHandler(req, res, next) {
  try {
    const games = await syncGames();
    res.json({ data: games });
  } catch (err) {
    next(err);
  }
}

async function getSiteSettingsHandler(req, res, next) {
  try {
    const settings = await getSettings();
    res.json({
      site_name: settings.siteName || '',
      site_description: settings.siteDescription || '',
      site_keywords: settings.siteKeywords || '',
      site_logo_url: settings.siteLogoUrl || '',
      site_favicon_url: settings.siteFaviconUrl || '',
      site_og_image_url: settings.siteOgImageUrl || '',
      cs_link: settings.csLink || '',
      footer_text: settings.footerText || '',
      maintenance_mode: settings.maintenanceMode || 'none',
    });
  } catch (err) {
    next(err);
  }
}

async function updateSiteSettingsHandler(req, res, next) {
  try {
    const payload = siteSchema.parse(req.body);
    const data = {};
    if (payload.site_name !== undefined) data.siteName = payload.site_name;
    if (payload.site_description !== undefined) data.siteDescription = payload.site_description;
    if (payload.site_keywords !== undefined) data.siteKeywords = payload.site_keywords;
    if (payload.site_logo_url !== undefined) data.siteLogoUrl = payload.site_logo_url;
    if (payload.site_favicon_url !== undefined) data.siteFaviconUrl = payload.site_favicon_url;
    if (payload.site_og_image_url !== undefined) data.siteOgImageUrl = payload.site_og_image_url;
    if (payload.cs_link !== undefined) data.csLink = payload.cs_link;
    if (payload.footer_text !== undefined) data.footerText = payload.footer_text;
    if (payload.maintenance_mode !== undefined) data.maintenanceMode = payload.maintenance_mode;
    const settings = await updateSiteSettings(data);
    res.json({
      site_name: settings.siteName || '',
      site_description: settings.siteDescription || '',
      site_keywords: settings.siteKeywords || '',
      site_logo_url: settings.siteLogoUrl || '',
      site_favicon_url: settings.siteFaviconUrl || '',
      site_og_image_url: settings.siteOgImageUrl || '',
      cs_link: settings.csLink || '',
      footer_text: settings.footerText || '',
      maintenance_mode: settings.maintenanceMode || 'none',
    });
  } catch (err) {
    next(err);
  }
}

async function publicSettingsHandler(req, res, next) {
  try {
    const data = await getPublicSettings();
    res.json(data);
  } catch (err) {
    next(err);
  }
}

async function listGamesHandler(req, res, next) {
  try {
    const games = await prisma.game.findMany({ orderBy: { createdAt: 'desc' } });
    res.json({ data: games });
  } catch (err) {
    next(err);
  }
}

async function updateGameStatusHandler(req, res, next) {
  try {
    const payload = gameStatusSchema.parse(req.body);
    const code = req.params.code;
    const game = await prisma.game.update({
      where: { code },
      data: { enabled: payload.enabled },
    });
    res.json({ data: game });
  } catch (err) {
    next(err);
  }
}

async function listTransactionsHandler(req, res, next) {
  try {
    const [orders, walletTransactions] = await Promise.all([
      prisma.topupOrder.findMany({
        include: { user: { select: { id: true, name: true, email: true } } },
        orderBy: { createdAt: 'desc' },
      }),
      prisma.walletTransaction.findMany({
        include: { user: { select: { id: true, name: true, email: true } } },
        orderBy: { createdAt: 'desc' },
      }),
    ]);
    res.json({ orders, wallet_transactions: walletTransactions });
  } catch (err) {
    next(err);
  }
}

function sumNumber(list, accessor) {
  return list.reduce((acc, item) => acc + Number(accessor(item) || 0), 0);
}

async function profitReportHandler(req, res, next) {
  try {
    const [orders, walletTransactions] = await Promise.all([
      prisma.topupOrder.findMany({}),
      prisma.walletTransaction.findMany({}),
    ]);

    const successOrders = orders.filter((order) => order.status === 'SUCCESS');
    const orderMargin = sumNumber(successOrders, (order) => Number(order.priceSell) - Number(order.priceApi));
    const orderFees = sumNumber(successOrders, (order) => order.fee);
    const depositFees = sumNumber(
      walletTransactions.filter((tx) => tx.type === 'DEPOSIT' && tx.status === 'PAID'),
      (tx) => tx.fee
    );

    const totalProfit = orderMargin + orderFees + depositFees;

    res.json({
      summary: {
        total_orders: orders.length,
        success_orders: successOrders.length,
        failed_orders: orders.filter((o) => o.status === 'FAILED').length,
        pending_orders: orders.filter((o) => o.status === 'PENDING_PAYMENT' || o.status === 'PROCESSING' || o.status === 'PAID').length,
      },
      profit: {
        order_margin: orderMargin,
        order_fee: orderFees,
        deposit_fee: depositFees,
        total: totalProfit,
      },
    });
  } catch (err) {
    next(err);
  }
}

module.exports = {
  login,
  getSettingsHandler,
  updateSettingsHandler,
  syncGamesHandler,
  listGamesHandler,
  updateGameStatusHandler,
  listTransactionsHandler,
  profitReportHandler,
  getSiteSettingsHandler,
  updateSiteSettingsHandler,
  publicSettingsHandler,
};
