const { listGames, getGameByCode } = require('../services/gameService');
const productService = require('../services/productService');
const { getSettings } = require('../services/settingsService');
const { applyPercent, toDecimal } = require('../utils/money');

function normalizeProducts(products, settings) {
  const marginPercent = settings.productMarginPercent || 0;
  return products.map((product) => {
    const priceApi = toDecimal(product.price || 0);
    const margin = applyPercent(priceApi, marginPercent);
    const priceSell = priceApi.add(margin);
    return {
      product_code: product.code,
      product_name: product.name,
      price_api: priceApi,
      price_sell: priceSell,
      icon_url: product.iconUrl,
    };
  });
}

async function list(req, res, next) {
  try {
    const games = await listGames();
    res.json({ data: games });
  } catch (err) {
    next(err);
  }
}

async function detail(req, res, next) {
  try {
    const game = await getGameByCode(req.params.code);
    if (!game) {
      return res.status(404).json({ message: 'Game not found' });
    }
    res.json({ data: game });
  } catch (err) {
    next(err);
  }
}

async function products(req, res, next) {
  try {
    const gameCode = req.params.code || req.query.game;
    if (!gameCode) {
      return res.status(400).json({ message: 'game is required' });
    }
    const settings = await getSettings();
    const products = await productService.listProducts(gameCode);
    const normalized = normalizeProducts(products, settings);
    res.json({ data: normalized });
  } catch (err) {
    next(err);
  }
}

async function gameDetail(req, res, next) {
  try {
    const gameCode = req.params.code;
    const game = await getGameByCode(gameCode);
    if (!game) {
      return res.status(404).json({ message: 'Game not found' });
    }
    const settings = await getSettings();
    const products = await productService.listProducts(gameCode);
    const normalized = normalizeProducts(products, settings);
    res.json({
      game_code: game.code,
      game_name: game.name,
      icon_url: game.iconUrl,
      input_fields: game.inputFields,
      products: normalized,
    });
  } catch (err) {
    next(err);
  }
}

module.exports = {
  list,
  detail,
  products,
  gameDetail,
};
