const path = require('path');
const { getSettings } = require('../services/settingsService');

let cachedMode = { value: null, expiresAt: 0 };

async function getMode() {
  const now = Date.now();
  if (cachedMode.value && cachedMode.expiresAt > now) {
    return cachedMode.value;
  }
  try {
    const settings = await getSettings();
    const mode = (settings.maintenanceMode || process.env.MAINTENANCE_MODE || 'none').toLowerCase().trim();
    cachedMode = { value: mode, expiresAt: now + 3000 };
    return mode;
  } catch (err) {
    const fallback = (process.env.MAINTENANCE_MODE || 'none').toLowerCase().trim();
    cachedMode = { value: fallback, expiresAt: now + 3000 };
    return fallback;
  }
}

function getDevToken() {
  return (process.env.DEVELOPER_TOKEN || '').trim();
}

function isBypass(req) {
  const token = req.headers['x-developer-token'] || req.query?.dev_token || '';
  const expected = getDevToken();
  return expected && token === expected;
}

function isAllowedStatic(pathname) {
  return pathname.startsWith('/assets')
    || pathname.startsWith('/images')
    || pathname.startsWith('/favicon')
    || pathname.endsWith('.css')
    || pathname.endsWith('.js');
}

function respondMaintenance(req, res) {
  if (req.path.startsWith('/api')) {
    return res.status(503).json({ message: 'Maintenance mode' });
  }
  const maintenancePage = path.join(__dirname, '..', '..', 'public', 'maintenance.html');
  return res.status(503).sendFile(maintenancePage);
}

async function maintenanceGate(req, res, next) {
  const mode = await getMode();
  if (mode === 'none') {
    return next();
  }

  if (isBypass(req)) {
    return next();
  }

  const pathname = req.path || '';
  if (isAllowedStatic(pathname)) {
    return next();
  }

  if (mode === 'developer') {
    return respondMaintenance(req, res);
  }

  if (mode === 'admin') {
    if (pathname.startsWith('/api/admin')) return next();
    if (pathname === '/admin.html') return next();
    return respondMaintenance(req, res);
  }

  return respondMaintenance(req, res);
}

module.exports = { maintenanceGate };
