const axios = require('axios');
const { createError } = require('../utils/errors');

function getConfig() {
  const baseUrl = process.env.CASHIFY_BASE_URL || 'https://cashify.my.id';
  const licenseKey = process.env.CASHIFY_KEY;
  if (!licenseKey) {
    throw createError(500, 'CASHIFY_KEY is not configured');
  }
  return { baseUrl, licenseKey };
}

async function createQris(amount) {
  const { baseUrl, licenseKey } = getConfig();
  const qrisId = process.env.CASHIFY_QRIS_ID;
  if (!qrisId) {
    throw createError(500, 'CASHIFY_QRIS_ID is not configured');
  }
  const res = await axios.post(`${baseUrl}/api/generate/qris`, {
    id: qrisId,
    amount,
    useUniqueCode: true,
    packageIds: [process.env.CASHIFY_PACKAGE_ID || 'com.orderkuota.app'],
    expiredInMinutes: 15,
  }, {
    headers: {
      'x-license-key': licenseKey,
      'Content-Type': 'application/json',
    },
  });
  return res.data;
}

async function createQrisV2(amount) {
  const { baseUrl, licenseKey } = getConfig();
  const qrisId = process.env.CASHIFY_QRIS_ID;
  if (!qrisId) {
    throw createError(500, 'CASHIFY_QRIS_ID is not configured');
  }
  const res = await axios.post(`${baseUrl}/api/generate/v2/qris`, {
    qr_id: qrisId,
    amount,
    useUniqueCode: true,
    packageIds: [process.env.CASHIFY_PACKAGE_ID || 'com.orderkuota.app'],
    expiredInMinutes: 15,
    qrType: 'dynamic',
    paymentMethod: 'qris',
    useQris: true,
  }, {
    headers: {
      'x-license-key': licenseKey,
      'Content-Type': 'application/json',
    },
  });
  return res.data;
}

async function payEwallet(amount) {
  const { baseUrl, licenseKey } = getConfig();
  const packageId = process.env.CASHIFY_EWALLET_PACKAGE_ID || 'id.dana';
  const res = await axios.post(`${baseUrl}/api/generate/v2/qris`, {
    amount,
    useUniqueCode: true,
    packageIds: [packageId],
    expiredInMinutes: 15,
    paymentMethod: 'ewallet',
    useQris: false,
  }, {
    headers: {
      'x-license-key': licenseKey,
      'Content-Type': 'application/json',
    },
  });
  return res.data;
}

async function checkStatus(transactionId) {
  const { baseUrl, licenseKey } = getConfig();
  const res = await axios.post(`${baseUrl}/api/generate/check-status`, {
    transactionId,
  }, {
    headers: {
      'x-license-key': licenseKey,
      'Content-Type': 'application/json',
    },
  });
  return res.data;
}

async function cancelStatus(transactionId) {
  const { baseUrl, licenseKey } = getConfig();
  const res = await axios.post(`${baseUrl}/api/generate/cancel-status`, {
    transactionId,
  }, {
    headers: {
      'x-license-key': licenseKey,
      'Content-Type': 'application/json',
    },
  });
  return res.data;
}

module.exports = {
  createQris,
  createQrisV2,
  payEwallet,
  checkStatus,
  cancelStatus,
};
