const prisma = require('../prisma');
const cashifyService = require('./cashifyService');
const { getSettings } = require('./settingsService');
const { applyPercent, toDecimal } = require('../utils/money');
const { createError } = require('../utils/errors');
const walletService = require('./walletService');

function extractTransactionId(payload) {
  return payload?.transactionId || payload?.data?.transactionId || payload?.data?.transaction_id || payload?.data?.id || null;
}

function extractPaymentMeta(payload) {
  return {
    raw: payload,
    qrisString: payload?.data?.qris_string || payload?.data?.qr_string || payload?.data?.qrcode || payload?.qris_string,
    qrisUrl: payload?.data?.qr_url || payload?.data?.qr_url_string || payload?.qr_url,
    expiredAt: payload?.data?.expired_at || payload?.data?.expiredAt || payload?.data?.expired_time || null,
  };
}

async function createDepositPayment({ user, amount, method }) {
  const settings = await getSettings();
  const baseAmount = toDecimal(amount);
  if (baseAmount.lte(0)) {
    throw createError(400, 'Amount must be greater than 0');
  }
  const fee = applyPercent(baseAmount, settings.depositFeePercent || 0);
  const totalPay = baseAmount.add(fee);

  const paymentPayload = method === 'QRIS'
    ? await cashifyService.createQrisV2(totalPay.toNumber())
    : await cashifyService.payEwallet(totalPay.toNumber());

  const paymentId = extractTransactionId(paymentPayload);
  const paymentMeta = extractPaymentMeta(paymentPayload);

  await prisma.walletTransaction.create({
    data: {
      walletId: user.wallet.id,
      userId: user.id,
      type: 'DEPOSIT',
      status: 'PENDING',
      amount: baseAmount,
      fee,
      total: totalPay,
      paymentMethod: method,
      referenceId: paymentId,
      description: 'Deposit saldo',
      meta: paymentMeta,
    },
  });

  return {
    payment_id: paymentId,
    amount: Number(baseAmount),
    fee: Number(fee),
    total: Number(totalPay),
    meta: paymentMeta,
  };
}

async function handleDepositStatus(paymentId) {
  const paymentStatus = await cashifyService.checkStatus(paymentId);
  const status = (paymentStatus?.data?.status || paymentStatus?.status || '').toString().toUpperCase();

  const transaction = await prisma.walletTransaction.findFirst({ where: { referenceId: paymentId } });
  if (!transaction) {
    return { status, updated: false };
  }

  if (status === 'PAID' && transaction.status === 'PENDING') {
    await prisma.$transaction(async (tx) => {
      await tx.walletTransaction.update({
        where: { id: transaction.id },
        data: { status: 'PAID' },
      });
      await walletService.adjustWalletBalance(tx, transaction.walletId, transaction.amount);
    });
    return { status, updated: true };
  }

  if (status === 'EXPIRED' && transaction.status === 'PENDING') {
    await prisma.walletTransaction.update({
      where: { id: transaction.id },
      data: { status: 'EXPIRED' },
    });
  }

  if ((status === 'CANCELLED' || status === 'CANCELED') && transaction.status === 'PENDING') {
    await prisma.walletTransaction.update({
      where: { id: transaction.id },
      data: { status: 'CANCELLED' },
    });
  }

  return { status, updated: false };
}

async function cancelDepositPayment(paymentId) {
  const result = await cashifyService.cancelStatus(paymentId);
  const status = (result?.data?.status || result?.status || '').toString().toUpperCase();
  const transaction = await prisma.walletTransaction.findFirst({ where: { referenceId: paymentId } });
  if (transaction && transaction.status === 'PENDING') {
    const nextStatus = (status === 'CANCELLED' || status === 'CANCELED') ? 'CANCELLED' : 'FAILED';
    await prisma.walletTransaction.update({
      where: { id: transaction.id },
      data: { status: nextStatus },
    });
  }
  const order = await prisma.topupOrder.findFirst({ where: { paymentId } });
  if (order && order.status === 'PENDING_PAYMENT') {
    const nextStatus = (status === 'CANCELLED' || status === 'CANCELED') ? 'CANCELLED' : 'FAILED';
    await prisma.topupOrder.update({
      where: { id: order.id },
      data: { status: nextStatus },
    });
  }
  return result;
}

module.exports = {
  createDepositPayment,
  handleDepositStatus,
  cancelDepositPayment,
};
