const prisma = require('../prisma');
const { hashPassword, verifyPassword } = require('../utils/crypto');
const { createError } = require('../utils/errors');
const crypto = require('crypto');

async function createUser({ name, email, phone, password }) {
  const existing = await prisma.user.findUnique({ where: { email } });
  if (existing) {
    throw createError(409, 'Email already registered');
  }
  const passwordHash = await hashPassword(password);
  return prisma.$transaction(async (tx) => {
    const user = await tx.user.create({
      data: {
        name,
        email,
        phone: phone || null,
        passwordHash,
      },
    });
    await tx.wallet.create({
      data: {
        userId: user.id,
        balance: 0,
      },
    });
    return user;
  });
}

async function findOrCreateGoogleUser({ name, email }) {
  const existing = await prisma.user.findUnique({ where: { email } });
  if (existing) {
    return existing;
  }
  const randomPassword = crypto.randomBytes(32).toString('hex');
  const passwordHash = await hashPassword(randomPassword);
  return prisma.$transaction(async (tx) => {
    const user = await tx.user.create({
      data: {
        name: name || email,
        email,
        passwordHash,
      },
    });
    await tx.wallet.create({
      data: {
        userId: user.id,
        balance: 0,
      },
    });
    return user;
  });
}

async function verifyUser(email, password) {
  const user = await prisma.user.findUnique({ where: { email } });
  if (!user) {
    throw createError(401, 'Invalid credentials');
  }
  const ok = await verifyPassword(password, user.passwordHash);
  if (!ok) {
    throw createError(401, 'Invalid credentials');
  }
  return user;
}

async function getUserById(id) {
  return prisma.user.findUnique({
    where: { id },
    include: { wallet: true },
  });
}

async function updateUser(id, data) {
  return prisma.user.update({
    where: { id },
    data: {
      name: data.name,
      email: data.email,
      phone: data.phone,
    },
  });
}

async function updatePassword(id, currentPassword, newPassword) {
  const user = await prisma.user.findUnique({ where: { id } });
  if (!user) {
    throw createError(404, 'User not found');
  }
  const ok = await verifyPassword(currentPassword, user.passwordHash);
  if (!ok) {
    throw createError(400, 'Current password is incorrect');
  }
  const passwordHash = await hashPassword(newPassword);
  await prisma.user.update({
    where: { id },
    data: { passwordHash },
  });
}

async function setPassword(id, newPassword) {
  const user = await prisma.user.findUnique({ where: { id } });
  if (!user) {
    throw createError(404, 'User not found');
  }
  const passwordHash = await hashPassword(newPassword);
  await prisma.user.update({
    where: { id },
    data: { passwordHash },
  });
}

module.exports = {
  createUser,
  findOrCreateGoogleUser,
  verifyUser,
  getUserById,
  updateUser,
  updatePassword,
  setPassword,
};
