const prisma = require('../prisma');
const { createError } = require('../utils/errors');
const { toDecimal } = require('../utils/money');

async function getWalletByUserId(userId) {
  const wallet = await prisma.wallet.findUnique({ where: { userId } });
  if (!wallet) {
    throw createError(404, 'Wallet not found');
  }
  return wallet;
}

async function adjustWalletBalance(tx, walletId, amount) {
  return tx.wallet.update({
    where: { id: walletId },
    data: {
      balance: {
        increment: amount,
      },
    },
  });
}

async function createWalletTransaction(tx, data) {
  return tx.walletTransaction.create({
    data,
  });
}

async function ensureSufficientBalance(walletId, amount) {
  const wallet = await prisma.wallet.findUnique({ where: { id: walletId } });
  if (!wallet) {
    throw createError(404, 'Wallet not found');
  }
  if (wallet.balance.lt(toDecimal(amount))) {
    throw createError(400, 'Insufficient balance');
  }
  return wallet;
}

module.exports = {
  getWalletByUserId,
  adjustWalletBalance,
  createWalletTransaction,
  ensureSufficientBalance,
};
