(() => {
  const ADMIN_TOKEN_KEY = 'admin_token';

  const loginCard = document.getElementById('adminLoginCard');
  const loginForm = document.getElementById('adminLoginForm');
  const loginMessage = document.getElementById('adminLoginMessage');
  const logoutBtn = document.getElementById('adminLogoutBtn');

  const settingsCard = document.getElementById('adminSettingsCard');
  const settingsForm = document.getElementById('adminSettingsForm');
  const settingsMessage = document.getElementById('adminSettingsMessage');
  const syncBtn = document.getElementById('syncGamesBtn');
  const gameList = document.getElementById('adminGameList');
  const siteCard = document.getElementById('adminSiteCard');
  const siteForm = document.getElementById('adminSiteForm');
  const siteMessage = document.getElementById('adminSiteMessage');

  const reportCard = document.getElementById('adminReportCard');
  const profitMargin = document.getElementById('profitMargin');
  const profitOrderFee = document.getElementById('profitOrderFee');
  const profitDepositFee = document.getElementById('profitDepositFee');
  const profitTotal = document.getElementById('profitTotal');
  const profitBarMargin = document.getElementById('profitBarMargin');
  const profitBarOrderFee = document.getElementById('profitBarOrderFee');
  const profitBarDepositFee = document.getElementById('profitBarDepositFee');

  const transactionsCard = document.getElementById('adminTransactionsCard');
  const orderCount = document.getElementById('orderCount');
  const walletCount = document.getElementById('walletCount');
  const userCount = document.getElementById('userCount');
  const txnList = document.getElementById('adminTxnList');
  const filterFromDate = document.getElementById('filterFromDate');
  const filterToDate = document.getElementById('filterToDate');
  const filterType = document.getElementById('filterType');
  const filterStatus = document.getElementById('filterStatus');
  const filterSearch = document.getElementById('filterSearch');
  const exportCsvBtn = document.getElementById('exportCsvBtn');
  const refreshTxnBtn = document.getElementById('refreshTxnBtn');
  const autoRefreshToggle = document.getElementById('autoRefreshToggle');
  const autoRefreshInterval = document.getElementById('autoRefreshInterval');
  const prevPageBtn = document.getElementById('prevPageBtn');
  const nextPageBtn = document.getElementById('nextPageBtn');
  const pageInfo = document.getElementById('pageInfo');

  const detailModal = document.getElementById('txnDetailModal');
  const detailType = document.getElementById('detailType');
  const detailStatus = document.getElementById('detailStatus');
  const detailAmount = document.getElementById('detailAmount');
  const detailRef = document.getElementById('detailRef');
  const detailUser = document.getElementById('detailUser');
  const detailProduct = document.getElementById('detailProduct');
  const detailGame = document.getElementById('detailGame');
  const detailMethod = document.getElementById('detailMethod');
  const detailTime = document.getElementById('detailTime');

  let cachedTransactions = [];
  let currentPage = 1;
  const pageSize = 25;
  let refreshTimer = null;

  function getAdminToken() {
    return localStorage.getItem(ADMIN_TOKEN_KEY);
  }

  function setAdminToken(token) {
    if (token) localStorage.setItem(ADMIN_TOKEN_KEY, token);
  }

  function clearAdminToken() {
    localStorage.removeItem(ADMIN_TOKEN_KEY);
  }

  async function adminFetch(path, options = {}) {
    const headers = Object.assign({ 'Content-Type': 'application/json' }, options.headers || {});
    const token = getAdminToken();
    if (token) {
      headers.Authorization = `Bearer ${token}`;
    }
    if (typeof getDevToken === 'function') {
      const devToken = getDevToken();
      if (devToken) {
        headers['X-Developer-Token'] = devToken;
      }
    }
    const res = await fetch(`${API_BASE}${path}`, { ...options, headers });
    const contentType = res.headers.get('content-type') || '';
    const body = contentType.includes('application/json') ? await res.json() : await res.text();
    if (!res.ok) {
      const message = body && body.message ? body.message : (body.error || 'Request failed');
      throw new Error(message);
    }
    return body;
  }

  function showPanel(active) {
    if (active) {
      loginCard?.classList.add('d-none');
      settingsCard?.classList.remove('d-none');
      siteCard?.classList.remove('d-none');
      reportCard?.classList.remove('d-none');
      transactionsCard?.classList.remove('d-none');
      document.getElementById('adminGamesCard')?.classList.remove('d-none');
    } else {
      loginCard?.classList.remove('d-none');
      settingsCard?.classList.add('d-none');
      siteCard?.classList.add('d-none');
      reportCard?.classList.add('d-none');
      transactionsCard?.classList.add('d-none');
      document.getElementById('adminGamesCard')?.classList.add('d-none');
    }
  }

  async function loadSettings() {
    const data = await adminFetch('/api/admin/settings');
    if (!settingsForm) return;
    settingsForm.querySelector('[name="product_margin_percent"]').value = data.product_margin_percent ?? 0;
    settingsForm.querySelector('[name="deposit_fee_percent"]').value = data.deposit_fee_percent ?? 0;
  }

  async function loadGames() {
    if (!gameList) return;
    gameList.innerHTML = '<tr><td colspan="3" class="muted">Loading...</td></tr>';
    const data = await adminFetch('/api/admin/games');
    const games = data.data || [];
    if (!games.length) {
      gameList.innerHTML = '<tr><td colspan="3" class="muted">Tidak ada game.</td></tr>';
      return;
    }
    gameList.innerHTML = '';
    games.forEach((game) => {
      const row = document.createElement('tr');
      row.innerHTML = `
        <td>${game.code}</td>
        <td>${game.name}</td>
        <td>
          <div class="form-check form-switch">
            <input class="form-check-input" type="checkbox" ${game.enabled ? 'checked' : ''}>
          </div>
        </td>
      `;
      const checkbox = row.querySelector('input[type="checkbox"]');
      checkbox.addEventListener('change', async () => {
        checkbox.disabled = true;
        try {
          await adminFetch(`/api/admin/games/${encodeURIComponent(game.code)}`, {
            method: 'PATCH',
            body: JSON.stringify({ enabled: checkbox.checked }),
          });
        } catch (err) {
          checkbox.checked = !checkbox.checked;
          alert(err.message || 'Gagal mengubah status game.');
        } finally {
          checkbox.disabled = false;
        }
      });
      gameList.appendChild(row);
    });
  }

  async function loadSiteSettings() {
    if (!siteForm) return;
    const data = await adminFetch('/api/admin/site');
    siteForm.querySelector('[name="site_name"]').value = data.site_name || '';
    siteForm.querySelector('[name="site_description"]').value = data.site_description || '';
    siteForm.querySelector('[name="site_keywords"]').value = data.site_keywords || '';
    siteForm.querySelector('[name="site_logo_url"]').value = data.site_logo_url || '';
    siteForm.querySelector('[name="site_favicon_url"]').value = data.site_favicon_url || '';
    siteForm.querySelector('[name="site_og_image_url"]').value = data.site_og_image_url || '';
    siteForm.querySelector('[name="cs_link"]').value = data.cs_link || '';
    siteForm.querySelector('[name="footer_text"]').value = data.footer_text || '';
    siteForm.querySelector('[name="maintenance_mode"]').value = data.maintenance_mode || 'none';
  }

  async function loadReports() {
    const report = await adminFetch('/api/admin/reports/profit');
    if (profitMargin) profitMargin.textContent = formatRupiah(report.profit?.order_margin || 0);
    if (profitOrderFee) profitOrderFee.textContent = formatRupiah(report.profit?.order_fee || 0);
    if (profitDepositFee) profitDepositFee.textContent = formatRupiah(report.profit?.deposit_fee || 0);
    if (profitTotal) profitTotal.textContent = formatRupiah(report.profit?.total || 0);
    const total = Number(report.profit?.total || 0) || 0;
    const marginPct = total ? (Number(report.profit?.order_margin || 0) / total) * 100 : 0;
    const orderFeePct = total ? (Number(report.profit?.order_fee || 0) / total) * 100 : 0;
    const depositFeePct = total ? (Number(report.profit?.deposit_fee || 0) / total) * 100 : 0;
    if (profitBarMargin) profitBarMargin.style.width = `${marginPct.toFixed(2)}%`;
    if (profitBarOrderFee) profitBarOrderFee.style.width = `${orderFeePct.toFixed(2)}%`;
    if (profitBarDepositFee) profitBarDepositFee.style.width = `${depositFeePct.toFixed(2)}%`;
  }

  async function loadTransactions() {
    const data = await adminFetch('/api/admin/transactions');
    const orders = data.orders || [];
    const walletTx = data.wallet_transactions || [];
    if (orderCount) orderCount.textContent = orders.length;
    if (walletCount) walletCount.textContent = walletTx.length;
    const uniqueUsers = new Set([
      ...orders.map((o) => o.userId),
      ...walletTx.map((t) => t.userId),
    ]);
    if (userCount) userCount.textContent = uniqueUsers.size;

    cachedTransactions = [
      ...orders.map((o) => ({
        createdAt: o.createdAt,
        type: 'ORDER',
        status: o.status,
        amount: o.totalPay,
        ref: o.id,
        user: o.user?.email || o.user?.name || o.userId,
        product: o.productName,
        game: o.gameCode,
        method: o.paymentMethod,
      })),
      ...walletTx.map((t) => ({
        createdAt: t.createdAt,
        type: t.type,
        status: t.status,
        amount: t.total || t.amount,
        ref: t.referenceId || t.id,
        user: t.user?.email || t.user?.name || t.userId,
        product: t.description,
        game: '-',
        method: t.paymentMethod,
      })),
    ].sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

    renderTransactions();
  }

  function getFilteredTransactions() {
    let list = [...cachedTransactions];

    const fromValue = filterFromDate?.value;
    const toValue = filterToDate?.value;
    if (fromValue) {
      const fromDate = new Date(`${fromValue}T00:00:00`);
      list = list.filter((item) => new Date(item.createdAt) >= fromDate);
    }
    if (toValue) {
      const toDate = new Date(`${toValue}T23:59:59`);
      list = list.filter((item) => new Date(item.createdAt) <= toDate);
    }

    const typeValue = filterType?.value?.trim().toUpperCase();
    if (typeValue) {
      list = list.filter((item) => item.type === typeValue);
    }

    const statusValue = filterStatus?.value?.trim().toUpperCase();
    if (statusValue) {
      list = list.filter((item) => (item.status || '').toUpperCase() === statusValue);
    }

    const term = filterSearch?.value?.trim().toLowerCase();
    if (term) {
      list = list.filter((item) => {
        const haystack = `${item.type} ${item.status} ${item.ref} ${item.user} ${item.product} ${item.game}`.toLowerCase();
        return haystack.includes(term);
      });
    }

    return list;
  }

  function renderTransactions() {
    if (!txnList) return;
    const list = getFilteredTransactions();

    if (!list.length) {
      txnList.innerHTML = '<tr><td colspan="5" class="muted">Belum ada transaksi.</td></tr>';
      if (pageInfo) pageInfo.textContent = 'Page 1';
      return;
    }

    txnList.innerHTML = '';
    const totalPages = Math.max(1, Math.ceil(list.length / pageSize));
    if (currentPage > totalPages) currentPage = totalPages;
    const start = (currentPage - 1) * pageSize;
    const pageItems = list.slice(start, start + pageSize);
    if (pageInfo) pageInfo.textContent = `Page ${currentPage} / ${totalPages}`;
    if (prevPageBtn) prevPageBtn.disabled = currentPage <= 1;
    if (nextPageBtn) nextPageBtn.disabled = currentPage >= totalPages;

    pageItems.forEach((item) => {
      const row = document.createElement('tr');
      row.innerHTML = `
        <td>${new Date(item.createdAt).toLocaleString('id-ID')}</td>
        <td>${item.type}</td>
        <td>${item.status}</td>
        <td>${formatRupiah(item.amount || 0)}</td>
        <td>${item.ref || '-'}</td>
      `;
      row.style.cursor = 'pointer';
      row.addEventListener('click', () => showDetail(item));
      txnList.appendChild(row);
    });
  }

  function showDetail(item) {
    if (detailType) detailType.textContent = item.type || '-';
    if (detailStatus) detailStatus.textContent = item.status || '-';
    if (detailAmount) detailAmount.textContent = formatRupiah(item.amount || 0);
    if (detailRef) detailRef.textContent = item.ref || '-';
    if (detailUser) detailUser.textContent = item.user || '-';
    if (detailProduct) detailProduct.textContent = item.product || '-';
    if (detailGame) detailGame.textContent = item.game || '-';
    if (detailMethod) detailMethod.textContent = item.method || '-';
    if (detailTime) detailTime.textContent = new Date(item.createdAt).toLocaleString('id-ID');

    if (detailModal && window.bootstrap?.Modal) {
      const modal = window.bootstrap.Modal.getOrCreateInstance(detailModal);
      modal.show();
    }
  }

  function exportCsv() {
    const list = getFilteredTransactions();
    if (!list.length) return;
    const rows = [
      ['Tanggal', 'Jenis', 'Status', 'Nominal', 'Ref', 'User', 'Produk', 'Game', 'Metode'],
      ...list.map((item) => [
        new Date(item.createdAt).toLocaleString('id-ID'),
        item.type,
        item.status,
        item.amount || 0,
        item.ref || '',
        item.user || '',
        item.product || '',
        item.game || '',
        item.method || '',
      ]),
    ];
    const csv = rows.map((row) => row.map((value) => `"${String(value).replace(/"/g, '""')}"`).join(',')).join('\n');
    const blob = new Blob([csv], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.href = url;
    link.download = `menvora-transactions-${Date.now()}.csv`;
    link.click();
    URL.revokeObjectURL(url);
  }

  async function initPanel() {
    try {
      await loadSettings();
      await loadSiteSettings();
      await loadGames();
      await loadReports();
      await loadTransactions();
    } catch (err) {
      if (err.message?.toLowerCase().includes('unauthorized')) {
        clearAdminToken();
        showPanel(false);
      }
    }
  }

  if (loginForm) {
    loginForm.addEventListener('submit', async (event) => {
      event.preventDefault();
      const email = loginForm.querySelector('[name="email"]').value.trim();
      const password = loginForm.querySelector('[name="password"]').value;
      if (!email || !password) return;
      try {
        const result = await adminFetch('/api/admin/login', {
          method: 'POST',
          body: JSON.stringify({ email, password }),
        });
        setAdminToken(result.token);
        loginMessage.textContent = 'Login berhasil.';
        showPanel(true);
        await initPanel();
      } catch (err) {
        loginMessage.textContent = err.message || 'Login gagal.';
      }
    });
  }

  if (settingsForm) {
    settingsForm.addEventListener('submit', async (event) => {
      event.preventDefault();
      const payload = {
        product_margin_percent: Number(settingsForm.querySelector('[name="product_margin_percent"]').value || 0),
        deposit_fee_percent: Number(settingsForm.querySelector('[name="deposit_fee_percent"]').value || 0),
      };
      try {
        await adminFetch('/api/admin/settings', {
          method: 'PUT',
          body: JSON.stringify(payload),
        });
        settingsMessage.textContent = 'Pengaturan tersimpan.';
      } catch (err) {
        settingsMessage.textContent = err.message || 'Gagal menyimpan.';
      }
    });
  }

  if (siteForm) {
    siteForm.addEventListener('submit', async (event) => {
      event.preventDefault();
      const payload = {
        site_name: siteForm.querySelector('[name="site_name"]').value.trim(),
        site_description: siteForm.querySelector('[name="site_description"]').value.trim(),
        site_keywords: siteForm.querySelector('[name="site_keywords"]').value.trim(),
        site_logo_url: siteForm.querySelector('[name="site_logo_url"]').value.trim(),
        site_favicon_url: siteForm.querySelector('[name="site_favicon_url"]').value.trim(),
        site_og_image_url: siteForm.querySelector('[name="site_og_image_url"]').value.trim(),
        cs_link: siteForm.querySelector('[name="cs_link"]').value.trim(),
        footer_text: siteForm.querySelector('[name="footer_text"]').value.trim(),
        maintenance_mode: siteForm.querySelector('[name="maintenance_mode"]').value,
      };
      try {
        await adminFetch('/api/admin/site', {
          method: 'PUT',
          body: JSON.stringify(payload),
        });
        if (siteMessage) siteMessage.textContent = 'Pengaturan tersimpan.';
      } catch (err) {
        if (siteMessage) siteMessage.textContent = err.message || 'Gagal menyimpan.';
      }
    });
  }

  if (syncBtn) {
    syncBtn.addEventListener('click', async () => {
      syncBtn.disabled = true;
      syncBtn.textContent = 'Syncing...';
      try {
        await adminFetch('/api/admin/games/sync', { method: 'POST' });
        await loadGames();
      } catch (err) {
        alert(err.message || 'Sync gagal.');
      } finally {
        syncBtn.disabled = false;
        syncBtn.textContent = 'Sync Game';
      }
    });
  }

  if (filterFromDate) {
    filterFromDate.addEventListener('change', renderTransactions);
  }
  if (filterToDate) {
    filterToDate.addEventListener('change', renderTransactions);
  }
  if (filterType) {
    filterType.addEventListener('change', renderTransactions);
  }
  if (filterStatus) {
    filterStatus.addEventListener('change', renderTransactions);
  }
  if (filterSearch) {
    filterSearch.addEventListener('input', renderTransactions);
  }
  if (exportCsvBtn) {
    exportCsvBtn.addEventListener('click', exportCsv);
  }
  if (refreshTxnBtn) {
    refreshTxnBtn.addEventListener('click', async () => {
      refreshTxnBtn.disabled = true;
      try {
        await loadTransactions();
        await loadReports();
      } catch (err) {
        alert(err.message || 'Gagal refresh data.');
      } finally {
        refreshTxnBtn.disabled = false;
      }
    });
  }
  if (prevPageBtn) {
    prevPageBtn.addEventListener('click', () => {
      if (currentPage > 1) {
        currentPage -= 1;
        renderTransactions();
      }
    });
  }
  if (nextPageBtn) {
    nextPageBtn.addEventListener('click', () => {
      currentPage += 1;
      renderTransactions();
    });
  }
  if (autoRefreshToggle) {
    autoRefreshToggle.addEventListener('change', () => {
      if (autoRefreshToggle.checked) {
        const interval = Number(autoRefreshInterval?.value || 60000);
        if (refreshTimer) clearInterval(refreshTimer);
        refreshTimer = setInterval(async () => {
          await loadTransactions();
          await loadReports();
        }, interval);
      } else if (refreshTimer) {
        clearInterval(refreshTimer);
        refreshTimer = null;
      }
    });
  }
  if (autoRefreshInterval) {
    autoRefreshInterval.addEventListener('change', () => {
      if (autoRefreshToggle?.checked) {
        if (refreshTimer) clearInterval(refreshTimer);
        const interval = Number(autoRefreshInterval.value || 60000);
        refreshTimer = setInterval(async () => {
          await loadTransactions();
          await loadReports();
        }, interval);
      }
    });
  }

  if (logoutBtn) {
    logoutBtn.addEventListener('click', () => {
      clearAdminToken();
      showPanel(false);
    });
  }

  const hasToken = !!getAdminToken();
  showPanel(hasToken);
  if (hasToken) {
    initPanel();
  }
})();
