const API_BASE = 'http://localhost:3000';
const TOKEN_KEY = 'token';
const CS_LINK = '';
const SITE_NAME = 'Menvora';
const SITE_DESCRIPTION = 'Platform top up game cepat, aman, dan otomatis.';
const SITE_KEYWORDS = 'topup game, voucher game, isi ulang, pulsa, ewallet, tagihan';
const SITE_LOGO_URL = 'assets/icons/placeholder.svg';
const SITE_FAVICON_URL = 'assets/icons/placeholder.svg';
const SITE_OG_IMAGE_URL = 'assets/icons/placeholder.svg';
const SITE_THEME_COLOR = '#2196F3';
const SITE_FOOTER_TEXT = `© ${new Date().getFullYear()} Menvora. All rights reserved.`;
const SITE_FOOTER_LINKS = [
  { label: 'Login', href: 'login.html' },
  { label: 'Register', href: 'register.html' },
];
const DEV_TOKEN_KEY = 'dev_token';

function getToken() {
  return localStorage.getItem(TOKEN_KEY);
}

function setToken(token) {
  if (token) {
    localStorage.setItem(TOKEN_KEY, token);
  }
}

function clearToken() {
  localStorage.removeItem(TOKEN_KEY);
}

function captureDevToken() {
  try {
    const params = new URLSearchParams(window.location.search);
    const token = params.get('dev_token');
    if (token) {
      sessionStorage.setItem(DEV_TOKEN_KEY, token);
      params.delete('dev_token');
      const cleanUrl = `${window.location.pathname}${params.toString() ? `?${params}` : ''}`;
      window.history.replaceState({}, document.title, cleanUrl);
    }
  } catch (err) {
    // ignore
  }
}

function getDevToken() {
  return sessionStorage.getItem(DEV_TOKEN_KEY);
}

captureDevToken();

async function loadSiteConfig() {
  try {
    const res = await fetch(`${API_BASE}/api/public/settings`);
    if (!res.ok) return null;
    const data = await res.json();
    window.__SITE_CONFIG__ = data;
    return data;
  } catch (err) {
    return null;
  }
}

window.__SITE_CONFIG_PROMISE__ = loadSiteConfig();

async function apiFetch(path, options = {}) {
  const headers = Object.assign({ 'Content-Type': 'application/json' }, options.headers || {});
  const token = getToken();
  if (token) {
    headers.Authorization = `Bearer ${token}`;
  }
  const devToken = getDevToken();
  if (devToken) {
    headers['X-Developer-Token'] = devToken;
  }
  const res = await fetch(`${API_BASE}${path}`, { ...options, headers });
  const contentType = res.headers.get('content-type') || '';
  const body = contentType.includes('application/json') ? await res.json() : await res.text();
  if (!res.ok) {
    const message = body && body.message ? body.message : (body.error || 'Request failed');
    throw new Error(message);
  }
  return body;
}

function formatRupiah(value) {
  const amount = Number(value || 0);
  return new Intl.NumberFormat('id-ID', {
    style: 'currency',
    currency: 'IDR',
    minimumFractionDigits: 0,
  }).format(amount).replace('IDR', 'Rp').replace('Rp', 'Rp ');
}

function requestAmount(options = {}) {
  const {
    title = 'Masukkan nominal deposit',
    hint = 'Nominal dalam angka saja',
    confirmText = 'Lanjutkan',
  } = options;

  return new Promise((resolve) => {
    const existing = document.getElementById('menvora-amount-modal');
    if (existing) existing.remove();

    const overlay = document.createElement('div');
    overlay.id = 'menvora-amount-modal';
    overlay.style.position = 'fixed';
    overlay.style.inset = '0';
    overlay.style.display = 'flex';
    overlay.style.alignItems = 'center';
    overlay.style.justifyContent = 'center';
    overlay.style.background = 'rgba(0, 0, 0, 0.55)';
    overlay.style.zIndex = '9999';

    const card = document.createElement('div');
    card.style.width = '320px';
    card.style.maxWidth = '90%';
    card.style.background = '#ffffff';
    card.style.borderRadius = '16px';
    card.style.padding = '18px';
    card.style.boxShadow = '0 12px 28px rgba(0, 0, 0, 0.25)';
    card.style.fontFamily = 'inherit';

    const heading = document.createElement('h3');
    heading.textContent = title;
    heading.style.fontSize = '16px';
    heading.style.margin = '0 0 6px 0';

    const sub = document.createElement('p');
    sub.textContent = hint;
    sub.style.fontSize = '12px';
    sub.style.color = '#6b7280';
    sub.style.margin = '0 0 12px 0';

    const input = document.createElement('input');
    input.type = 'number';
    input.inputMode = 'numeric';
    input.placeholder = 'Contoh: 50000';
    input.style.width = '100%';
    input.style.padding = '10px 12px';
    input.style.borderRadius = '10px';
    input.style.border = '1px solid #e5e7eb';
    input.style.outline = 'none';
    input.style.fontSize = '14px';

    const error = document.createElement('div');
    error.textContent = 'Nominal tidak valid.';
    error.style.fontSize = '12px';
    error.style.color = '#dc2626';
    error.style.marginTop = '6px';
    error.style.display = 'none';

    const actions = document.createElement('div');
    actions.style.display = 'flex';
    actions.style.gap = '10px';
    actions.style.marginTop = '14px';

    const cancelBtn = document.createElement('button');
    cancelBtn.type = 'button';
    cancelBtn.textContent = 'Batal';
    cancelBtn.style.flex = '1';
    cancelBtn.style.padding = '10px';
    cancelBtn.style.borderRadius = '10px';
    cancelBtn.style.border = '1px solid #e5e7eb';
    cancelBtn.style.background = '#f3f4f6';
    cancelBtn.style.cursor = 'pointer';

    const confirmBtn = document.createElement('button');
    confirmBtn.type = 'button';
    confirmBtn.textContent = confirmText;
    confirmBtn.style.flex = '1';
    confirmBtn.style.padding = '10px';
    confirmBtn.style.borderRadius = '10px';
    confirmBtn.style.border = 'none';
    confirmBtn.style.background = '#42A5F5';
    confirmBtn.style.color = '#ffffff';
    confirmBtn.style.cursor = 'pointer';

    const close = (value) => {
      overlay.remove();
      resolve(value);
    };

    const submit = () => {
      const amount = Number(input.value);
      if (!amount || Number.isNaN(amount)) {
        error.style.display = 'block';
        return;
      }
      close(amount);
    };

    overlay.addEventListener('click', (event) => {
      if (event.target === overlay) close(null);
    });
    cancelBtn.addEventListener('click', () => close(null));
    confirmBtn.addEventListener('click', submit);
    input.addEventListener('input', () => {
      error.style.display = 'none';
    });
    input.addEventListener('keydown', (event) => {
      if (event.key === 'Enter') submit();
    });

    card.appendChild(heading);
    card.appendChild(sub);
    card.appendChild(input);
    card.appendChild(error);
    actions.appendChild(cancelBtn);
    actions.appendChild(confirmBtn);
    card.appendChild(actions);
    overlay.appendChild(card);
    document.body.appendChild(overlay);
    input.focus();
  });
}
