(async function () {
  const token = getToken();
  if (!token) {
    window.location.href = 'login.html';
    return;
  }

  const params = new URLSearchParams(window.location.search);
  const gameCode = params.get('game');
  if (!gameCode) {
    window.location.href = 'game_list.html';
    return;
  }
  const defaultIcon = 'assets/icons/placeholder.svg';

  try {
    const data = await apiFetch(`/api/games/${encodeURIComponent(gameCode)}/detail`);
    const { game_name, icon_url, input_fields, products } = data;

    const textTop = document.getElementById('textTop');
    if (textTop) textTop.textContent = game_name || gameCode;
    const imgTop = document.getElementById('imgTop');
    if (imgTop) {
      imgTop.src = icon_url || defaultIcon;
      imgTop.classList.remove('u-bg', 'p-1');
    }

    const fieldList = Array.isArray(input_fields) ? input_fields : [];
    const primaryField = fieldList[0] || { name: 'user_id', label: 'User ID', type: 'text', required: true };
    const secondaryField = fieldList[1];
    const override = getInputOverride(gameCode, game_name);

    const primaryInput = document.getElementById('nomoId');
    if (primaryInput) {
      const baseLabel = primaryField.label || primaryField.name;
      const useOverride = override && (!baseLabel || baseLabel.toLowerCase() === 'user id' || baseLabel.toLowerCase() === 'user_id');
      const finalLabel = useOverride ? override.label : baseLabel;
      primaryInput.placeholder = finalLabel;
      primaryInput.dataset.field = primaryField.name;
      primaryInput.type = useOverride ? override.type : (primaryField.type || 'text');
    }

    const serverInput = document.getElementById('serverId');
    if (serverInput) {
      if (secondaryField) {
        serverInput.classList.remove('d-none');
        serverInput.placeholder = secondaryField.label || secondaryField.name;
        serverInput.dataset.field = secondaryField.name;
        serverInput.type = secondaryField.type || 'text';
      } else {
        serverInput.classList.add('d-none');
      }
    }

    const container = document.getElementById('dataProduk');
    if (!container) return;
    if (!products || !products.length) {
      container.innerHTML = '<p class="text-center mt-3">Produk belum tersedia.</p>';
      return;
    }

    const allProducts = products;
    let searchTerm = '';
    let sortMode = null;
    let walletBalance = null;

    try {
      const wallet = await apiFetch('/api/wallet/balance');
      walletBalance = Number(wallet.balance || 0);
    } catch (err) {
      walletBalance = null;
    }

    const searchInput = document.getElementById('search');
    if (searchInput) {
      searchInput.addEventListener('input', (event) => {
        searchTerm = event.target.value.trim().toLowerCase();
        applyFilters();
      });
    }

    window.rubahData = function rubahData(mode) {
      sortMode = mode;
      applyFilters();
    };

    window.rubahTampilan = function rubahTampilan() {};
    window.rubahGambar = function rubahGambar() {};
    window.rubahPencarian = function rubahPencarian() {};
    window.rubahSrcPil = function rubahSrcPil() {};

    function sortProducts(list, mode) {
      const copy = [...list];
      if (mode === 'A-Z') {
        return copy.sort((a, b) => String(a.product_name || '').localeCompare(String(b.product_name || '')));
      }
      if (mode === 'Z-A') {
        return copy.sort((a, b) => String(b.product_name || '').localeCompare(String(a.product_name || '')));
      }
      if (mode === 'Termurah') {
        return copy.sort((a, b) => Number(a.price_sell || 0) - Number(b.price_sell || 0));
      }
      if (mode === 'Termahal') {
        return copy.sort((a, b) => Number(b.price_sell || 0) - Number(a.price_sell || 0));
      }
      return copy;
    }

    function renderProducts(list) {
      container.innerHTML = '<div class="animate__animated animate__fadeIn d-grid-2" id="productList"></div>';
      const listEl = document.getElementById('productList');
      list.forEach((product) => {
        const card = document.createElement('div');
        card.className = 'bg-produk animate__animated animate__flipInX';
        card.innerHTML = `
          <div class="box-utama box-grid">
            <img src="${product.icon_url || icon_url || defaultIcon}" class="box-img borad-5 mb-3" width="40">
            <div class="text-start">
              <p class="fwb-6">${product.product_name}</p>
              <p class="fwb-6 op-7">${game_name || gameCode}</p>
              <p class="fwb-6 f-14 u-color">${formatRupiah(product.price_sell)}</p>
            </div>
          </div>
        `;
        card.addEventListener('click', () => showDetail(product));
        listEl.appendChild(card);
      });
    }

    function applyFilters() {
      let filtered = allProducts;
      if (searchTerm) {
        filtered = filtered.filter((product) => {
          const haystack = `${product.product_name || ''} ${product.product_code || ''}`.toLowerCase();
          return haystack.includes(searchTerm);
        });
      }
      filtered = sortProducts(filtered, sortMode);
      if (!filtered.length) {
        container.innerHTML = '<p class="text-center mt-3">Produk tidak ditemukan.</p>';
        return;
      }
      renderProducts(filtered);
    }

    applyFilters();

    function collectInputs() {
      const inputs = {};
      if (primaryInput) {
        inputs[primaryInput.dataset.field || 'user_id'] = primaryInput.value.trim();
      }
      if (secondaryField && serverInput) {
        inputs[serverInput.dataset.field || 'server'] = serverInput.value.trim();
      }
      return inputs;
    }

    function showDetail(product) {
      const inputs = collectInputs();
      const missingPrimary = primaryField.required && (!inputs[primaryInput.dataset.field || 'user_id']);
      const missingSecondary = secondaryField && secondaryField.required && (!inputs[serverInput.dataset.field || 'server']);

      let targetDisplay = '';
      if (missingPrimary || missingSecondary) {
        targetDisplay = '<span class="text-danger">Masukan data dahulu...</span>';
      } else {
        targetDisplay = Object.values(inputs).join(' ');
      }

      const body = document.getElementById('bodyCanv');
      if (!body) return;
      const total = formatRupiah(product.price_sell);

      const canPayWithWallet = walletBalance === null ? true : walletBalance >= Number(product.price_sell || 0);
      body.innerHTML = `
        <div class="text-center">
          <img src="${product.icon_url || icon_url || defaultIcon}" class="borad-5" width="50">
          <div class="mt-3">
            <p class="fwb-6">${product.product_name}</p>
            <p class="fwb-6 f-14 u-color mt-1">${total}</p>
          </div>
        </div>
        <div class="p-3 mt-3 mb-2 borad-5" style="background-color: #f6f8ff; border: 2px dotted #2196F3;">
          <p class="fwb-5 op-7">Deskripsi Produk :</p>
          <p class="fwb-6">${game_name || gameCode}</p>
        </div>
        <div class="d-flex justify-content-between py-2" style="border-bottom: dotted #ededed;">
          <p class="fwb-5 op-7">Tujuan :</p>
          <p class="fwb-6">${targetDisplay}</p>
        </div>
        <div class="d-flex justify-content-between py-2" style="border-bottom: dotted #ededed;">
          <p class="fwb-5 op-7">Nickname :</p>
          <p class="fwb-6">-</p>
        </div>
        <div class="py-2" style="border-bottom: dotted #ededed;">
          <p class="fwb-5 op-7">Total Pembayaran</p>
          <p class="fwb-6 f-14">${total}</p>
        </div>
        <div class="mt-3">
          <div class="d-flex align-items-center">
            <div class="w-100 bg-secondary p-2 borad-5" data-bs-dismiss="offcanvas" aria-label="Close">
              <p class="f-14 text-center text-white">Kembali</p>
            </div>
            <div class="w-100 u-bg p-2 borad-5 ms-3 ${canPayWithWallet ? '' : 'opacity-50'}" id="btnOrderWallet">
              <p class="f-14 text-center text-white">Bayar Saldo</p>
            </div>
          </div>
          <div class="d-flex align-items-center mt-2">
            <div class="w-100 border p-2 borad-5" id="btnOrderQris">
              <p class="f-14 text-center text-dark">Bayar QRIS</p>
            </div>
            <div class="w-100 border p-2 borad-5 ms-3" id="btnOrderEwallet">
              <p class="f-14 text-center text-dark">Bayar E-Wallet</p>
            </div>
          </div>
        </div>
      `;

      const detailCanvas = document.getElementById('detailCanv');
      if (detailCanvas && window.bootstrap && window.bootstrap.Offcanvas) {
        const offcanvas = window.bootstrap.Offcanvas.getOrCreateInstance(detailCanvas);
        offcanvas.show();
      }

      const btnOrderWallet = document.getElementById('btnOrderWallet');
      if (btnOrderWallet) {
        btnOrderWallet.addEventListener('click', async () => {
          if (!canPayWithWallet) {
            alert('Saldo tidak cukup. Silakan gunakan QRIS atau E-Wallet.');
            return;
          }
          const currentInputs = collectInputs();
          const primaryValue = currentInputs[primaryInput.dataset.field || 'user_id'];
          if (!primaryValue) {
            alert('Silakan isi data tujuan.');
            return;
          }
          try {
            await apiFetch('/api/orders', {
              method: 'POST',
              body: JSON.stringify({
                game_code: gameCode,
                product_code: product.product_code,
                payment_method: 'WALLET',
                inputs: currentInputs,
              }),
            });
            alert('Transaksi berhasil diproses.');
            window.location.href = 'transaksi.html';
          } catch (err) {
            alert(err.message || 'Transaksi gagal. Pastikan saldo cukup.');
          }
        });
      }

      async function handleExternalPayment(method) {
        const currentInputs = collectInputs();
        const primaryValue = currentInputs[primaryInput.dataset.field || 'user_id'];
        if (!primaryValue) {
          alert('Silakan isi data tujuan.');
          return;
        }
        try {
          const result = await apiFetch('/api/orders', {
            method: 'POST',
            body: JSON.stringify({
              game_code: gameCode,
              product_code: product.product_code,
              payment_method: method,
              inputs: currentInputs,
            }),
          });
          const payment = result.payment || result;
          const order = result.order || null;
          if (!payment?.payment_id) {
            throw new Error('Pembayaran tidak tersedia.');
          }
          const storageKey = `menvora_payment_${payment.payment_id}`;
          localStorage.setItem(storageKey, JSON.stringify({
            ...payment,
            type: 'ORDER',
            order_id: order?.id || null,
            method,
          }));
          window.location.href = `deposit.html?payment_id=${encodeURIComponent(payment.payment_id)}&method=${encodeURIComponent(method)}`;
        } catch (err) {
          alert(err.message || 'Gagal membuat pembayaran.');
        }
      }

      const btnOrderQris = document.getElementById('btnOrderQris');
      if (btnOrderQris) {
        btnOrderQris.addEventListener('click', () => handleExternalPayment('QRIS'));
      }

      const btnOrderEwallet = document.getElementById('btnOrderEwallet');
      if (btnOrderEwallet) {
        btnOrderEwallet.addEventListener('click', () => handleExternalPayment('EWALLET'));
      }
    }
  } catch (err) {
    console.error(err);
    alert(err.message || 'Gagal memuat produk.');
  }
})();

function getInputOverride(code, name) {
  const text = `${code || ''} ${name || ''}`.toLowerCase();
  if (/(pulsa|telkomsel|indosat|xl|axis|three|tri|smartfren)/.test(text)) {
    return { label: 'Nomor HP', type: 'tel' };
  }
  if (/(token|pln|listrik)/.test(text)) {
    return { label: 'ID Pelanggan / No Meter', type: 'text' };
  }
  if (/(bpjs)/.test(text)) {
    return { label: 'Nomor VA BPJS', type: 'text' };
  }
  if (/(pdam)/.test(text)) {
    return { label: 'ID Pelanggan PDAM', type: 'text' };
  }
  if (/(pbb)/.test(text)) {
    return { label: 'NOP PBB', type: 'text' };
  }
  if (/(internet|wifi|indihome|biznet|myrepublic)/.test(text)) {
    return { label: 'ID Pelanggan Internet', type: 'text' };
  }
  if (/(dana|ovo|gopay|linkaja|shopee|shopeepay|ewallet|e-wallet)/.test(text)) {
    return { label: 'Nomor HP / ID E-Wallet', type: 'tel' };
  }
  return null;
}

function getGaleri() {
  const input = document.getElementById('fileInput');
  if (input) {
    input.click();
  }
  return false;
}

async function previewImage(event) {
  const file = event?.target?.files?.[0];
  if (!file) return;

  const ocrInfo = document.getElementById('ocrInfo');
  const ocrResult = document.getElementById('ocrResult');
  if (ocrInfo) {
    ocrInfo.classList.remove('d-none');
    ocrInfo.textContent = 'Memeriksa gambar...';
  }
  if (ocrResult) {
    ocrResult.innerHTML = '';
  }

  if (!window.Tesseract) {
    if (ocrInfo) ocrInfo.textContent = 'OCR tidak tersedia.';
    return;
  }

  try {
    const result = await window.Tesseract.recognize(file, 'eng');
    const rawText = result?.data?.text || '';
    const cleaned = rawText.replace(/[^0-9\\n]/g, ' ').replace(/\\s+/g, ' ').trim();
    if (!cleaned) {
      if (ocrInfo) ocrInfo.textContent = 'Tidak ada angka terdeteksi.';
      return;
    }
    const values = Array.from(new Set(cleaned.split(' ').filter(Boolean))).slice(0, 6);
    if (ocrInfo) ocrInfo.textContent = 'Hasil OCR:';
    if (ocrResult) {
      ocrResult.innerHTML = '';
      values.forEach((val) => {
        const btn = document.createElement('button');
        btn.type = 'button';
        btn.className = 'pill-button';
        btn.textContent = val;
        btn.addEventListener('click', () => {
          const input = document.getElementById('nomoId');
          if (input) input.value = val;
        });
        ocrResult.appendChild(btn);
      });
    }
    const input = document.getElementById('nomoId');
    if (input && values[0]) {
      input.value = values[0];
    }
  } catch (err) {
    if (ocrInfo) ocrInfo.textContent = 'Gagal membaca gambar.';
  }
}
