(() => {
  function resolveConfig() {
    const runtime = window.__SITE_CONFIG__ || {};
    const siteName = (runtime.site_name || SITE_NAME || '').toString().trim();
    const description = (runtime.site_description || SITE_DESCRIPTION || '').toString().trim();
    const keywords = (runtime.site_keywords || SITE_KEYWORDS || '').toString().trim();
    const logoUrl = (runtime.site_logo_url || SITE_LOGO_URL || '').toString().trim();
    const faviconUrl = (runtime.site_favicon_url || SITE_FAVICON_URL || logoUrl || '').toString().trim();
    const ogImageUrl = (runtime.site_og_image_url || SITE_OG_IMAGE_URL || logoUrl || '').toString().trim();
    const themeColor = typeof SITE_THEME_COLOR === 'string' ? SITE_THEME_COLOR.trim() : '';
    const footerText = (runtime.footer_text || SITE_FOOTER_TEXT || '').toString().trim();
    const footerLinks = Array.isArray(SITE_FOOTER_LINKS) ? SITE_FOOTER_LINKS : [];
    return {
      siteName,
      description,
      keywords,
      logoUrl,
      faviconUrl,
      ogImageUrl,
      themeColor,
      footerText,
      footerLinks,
    };
  }

  function setTitle(siteName) {
    if (!siteName) return;
    const current = document.title || '';
    if (!current || /fast\s*pedia/i.test(current)) {
      document.title = siteName;
      return;
    }
    if (!current.toLowerCase().includes(siteName.toLowerCase())) {
      document.title = `${current} | ${siteName}`;
    }
  }

  function ensureMeta(name, content, attr = 'name') {
    if (!content) return;
    let meta = document.querySelector(`meta[${attr}="${name}"]`);
    if (!meta) {
      meta = document.createElement('meta');
      meta.setAttribute(attr, name);
      document.head.appendChild(meta);
    }
    meta.setAttribute('content', content);
  }

  function setFavicon(faviconUrl) {
    if (!faviconUrl) return;
    let link = document.querySelector('link[rel="icon"]');
    if (!link) {
      link = document.createElement('link');
      link.setAttribute('rel', 'icon');
      document.head.appendChild(link);
    }
    link.setAttribute('href', faviconUrl);
  }

  function applyLogo(logoUrl) {
    if (!logoUrl) return;
    document.querySelectorAll('img[data-site-logo], img#siteLogo').forEach((img) => {
      img.src = logoUrl;
    });
  }

  function applyFooter(footerText, footerLinks) {
    if (!footerText && !footerLinks.length) return;
    if (document.body?.dataset?.noFooter === 'true') return;
    if (document.getElementById('siteFooter')) return;

    const footer = document.createElement('footer');
    footer.id = 'siteFooter';
    footer.style.textAlign = 'center';
    footer.style.fontSize = '12px';
    footer.style.opacity = '0.7';
    footer.style.padding = '16px 12px 24px';

    const text = document.createElement('div');
    text.textContent = footerText;
    footer.appendChild(text);

    if (footerLinks.length) {
      const links = document.createElement('div');
      links.style.marginTop = '6px';
      footerLinks.forEach((link, index) => {
        if (!link || !link.href || !link.label) return;
        const anchor = document.createElement('a');
        anchor.href = link.href;
        anchor.textContent = link.label;
        anchor.style.margin = '0 6px';
        anchor.style.color = 'inherit';
        anchor.style.textDecoration = 'none';
        links.appendChild(anchor);
      });
      footer.appendChild(links);
    }

    document.body.appendChild(footer);
  }

  document.addEventListener('DOMContentLoaded', async () => {
    if (window.__SITE_CONFIG_PROMISE__) {
      try {
        await window.__SITE_CONFIG_PROMISE__;
      } catch (err) {
        // ignore
      }
    }
    const {
      siteName,
      description,
      keywords,
      logoUrl,
      faviconUrl,
      ogImageUrl,
      themeColor,
      footerText,
      footerLinks,
    } = resolveConfig();
    setTitle(siteName);
    ensureMeta('description', description);
    ensureMeta('keywords', keywords);
    ensureMeta('author', siteName);
    ensureMeta('theme-color', themeColor, 'name');
    ensureMeta('og:site_name', siteName, 'property');
    ensureMeta('og:type', 'website', 'property');
    ensureMeta('og:title', document.title || siteName, 'property');
    ensureMeta('og:description', description, 'property');
    ensureMeta('og:image', ogImageUrl, 'property');
    ensureMeta('twitter:card', 'summary_large_image');
    ensureMeta('twitter:title', document.title || siteName);
    ensureMeta('twitter:description', description);
    ensureMeta('twitter:image', ogImageUrl);
    setFavicon(faviconUrl);
    applyLogo(logoUrl);
    applyFooter(footerText, footerLinks);
  });
})();
